"""
 Script to perform bulk links insert/update/delete for Jabber Guest Links

 Required python modules:
    requests:  http://docs.python-requests.org/

 Usage: PROG [-h] -u USERNAME -p PASSWORD -s SERVER -f FILENAME [-t {XML,CSV}] -m {I,U,D,O}

  -h, --help    show this help message and exit
  -u USERNAME   Admin username
  -p PASSWORD   Admin password
  -s SERVER     JG server name or IP address
  -f FILENAME   Filename with links to insert
  -t {XML,CSV}  File format XML or CSV (optional, default to CSV)
  -m {I,U,D,O}  Insert, Update, Delete links, or Output to file

Copyright (C) 2014 - Cisco Systems
Author: David Nguyen (davidn@cisco.com)
Version: 1.0.2

Pursuant to the Jabber SDK License Agreement, this component of the SDK is Redistributable
when incorporated into a Developer Application."

"""

import requests
import json
import argparse
import csv
import xml.etree.ElementTree as ET
from xml.dom import minidom

# Initialize global variables
FIELDS = ['linkObjectId', 'requestPath', 'isEnabled', 'destination', 'displayName',
          'callerName', 'linkPrefix', 'useUniqueSipAlias', 'validAfter', 'validBefore',
          'callerAlias', 'userObjectId']

def get_links_from_jgs(url, session):
    'Get all links from Jabber Guest Server DB'

    resp = session.get(url)
    links = []
    if resp.status_code == 200:
        tree = ET.fromstring(resp.text)
        for link in tree.findall('Links'):
            alink = {}
            for node in link.iter():
                if node.tag != 'Links':
                    alink[node.tag] = node.text
                else:
                    links.append(alink)
    else:
        print "ERROR: get_links_from_jgs status =", resp.status_code
    return links

def get_links_from_file(filename, fileformat):
    'Read links from input file in XML or CSV format'

    links = []
    if fileformat == 'XML':
        tree = ET.parse(filename)
        root = tree.getroot()
        for link in root.findall('Links'):
            alink = {}
            for node in link.iter():
                if node.tag != 'Links':
                    alink[node.tag] = node.text
            links.append(alink)
    else:  # default is CSV format
        with open(filename, 'r') as f_handle:
            reader = csv.DictReader(f_handle)
            for row in reader:
                links.append(row)
    return links

class Links(object):
    'Class for jabber guest links object'

    def __init__(self):
        self._links = []  # Empty links list
        self.userobjectid = ''

    def get_userobjectid(self):
        'Get userObjectId for current user'

        return self.userobjectid

    def set_userobjectid(self, objid):
        'Assign userObjectId for links'

        self.userobjectid = objid

    def set_links(self, links):
        'Assign links to instance object'

        self._links = links

    def create_links(self, url, session):
        'Create new links on Jabber Guest Server'

        messages = {201: 'success',
                    400: 'failed to insert link',
                    500: 'link exists in DB!'}

        headers = {'content-type': 'application/json'}
        for link in self._links:
            print "Creating new links <" + link['displayName'] + ">"
            link['userObjectId'] = self.get_userobjectid()
            print link
            resp = session.post(url, data=json.dumps(link), headers=headers)
            if messages.has_key(resp.status_code):
                print "request status = %s => %s" % (resp.status_code, messages[resp.status_code])
            else:
                print "request status = ", resp.status_code

    def modify_links(self, url, session, db_links, mode):
        'Update or delete links'

        messages = {204: 'success',
                    404: 'failed to delete link'
        }
        headers = {'content-type': 'application/json'}

        for link in self._links:
            link_objid = ''
            # Find linkObjectId that matches requestPath from DB
            if link.has_key('linkObjectId') and link['linkObjectId'] != '':
                link_objid = link['linkObjectId']
            else:
                for db_link in db_links:
                    if link['requestPath'] == db_link['requestPath']:
                        link_objid = db_link['linkObjectId']
            if link_objid != '':
                if mode == 'U':
                    link['userObjectId'] = self.get_userobjectid()
                    print "Modifying link with requestPath= %s, linkObjectId=%s" %\
                          (link['requestPath'], link_objid)
                    resp = session.put(url+link_objid, data=json.dumps(link), headers=headers)
                else:
                    print "Deleting link with requestPath=%s, linkObjectId=%s" %\
                          (link['requestPath'], link_objid)
                    resp = session.delete(url+link_objid, data=json.dumps(link), headers=headers)
                if messages.has_key(resp.status_code):
                    print "request status = %s => %s" % (resp.status_code, messages[resp.status_code])
                else:
                    print "request status = ", resp.status_code
            else:
                print "Link with requestPath=%s doesn't exist, NO-OP" % (link['requestPath'])

    def validate_links(self):
        'Links validation'

        for link in self._links:
            for key in link.keys():
                if key not in FIELDS:
                    print "Invalid element found: ", key
                    return False
            # requestPath is a required field and must not be null
            if not link.has_key('requestPath') or link['requestPath'] == '':
                print "ERROR: Link must have required non-null field 'requestPath':", link
                return False
        return True

    def __str__(self):
        return str(self._links)


def get_userobjectid(session, url):
    'Returns userObjectId for request user.'

    resp = session.get(url, timeout=5)
    if resp.status_code == 200:
        users = resp.json()
        for key, value in users.items():
            if key == 'Users':
                for user in value:
                    if user['alias'] == session.auth[0]:
                        return user['objectId']
    else:
        print "Can't authenticate user", session.auth
        print "HTTP request status = ", resp.status_code
        return ''

def prettify(elem):
    'Return a pretty-printed XML string for the Element.'

    rough_string = ET.tostring(elem, 'utf-8')
    reparsed = minidom.parseString(rough_string)
    return reparsed.toprettyxml(indent="\t")

def output_to_file(db_links, filename, filetype):
    """
    Output all links from Jabber Guest Server to file with filename.
    This is useful for creating file to be used in update and delete links operation
    """

    print "Dumped links to file <%s> file type is <%s>" % (filename, filetype)
    with open(filename, 'w') as f_handle:
        if filetype == 'CSV':
            f_handle.write(','.join(FIELDS) + '\n')
            for link in db_links:
                line = []
                for field in FIELDS:
                    if link.has_key(field) and type(link.get(field)) is str:
                        line.append(link.get(field))
                    else:
                        line.append('')
                f_handle.write(','.join(line)+'\n')
        else:
            root = ET.Element('Links')
            root.set('total', str(len(db_links)))
            for link in db_links:
                top = ET.SubElement(root, 'Links')
                for key, val in link.items():
                    if type(val) is str:
                        child = ET.SubElement(top, key)
                        child.text = val
            f_handle.write(prettify(root))

def parse_args():
    'Parse command arguments'

    parser = argparse.ArgumentParser()
    parser.add_argument('-u', dest='username', help="Admin username", required=True)
    parser.add_argument('-p', dest='password', help="Admin password", required=True)
    parser.add_argument('-s', dest='server', help="JG server name or IP address", required=True)
    parser.add_argument('-f', dest='filename', help="Filename with links to insert", required=True)
    parser.add_argument('-t', dest='filetype', help="File format XML or CSV", default='CSV',
                        choices=['XML', 'CSV'])
    parser.add_argument('-m', dest='mode', help="Insert, Update, Delete links, or Output to file",
                        choices=['I', 'U', 'D', 'O'], required=True)
    args = parser.parse_args()
    return vars(args)

if __name__ == '__main__':
    cmdargs = parse_args()
    username, password, server, filename, filetype, mode = (
        cmdargs['username'], cmdargs['password'], cmdargs['server'],
        cmdargs['filename'], cmdargs['filetype'], cmdargs['mode'])
    linksurl = 'https://' + server + '/cjg-api/rest/links/'
    usersurl = 'https://' + server + '/cjg-api/rest/users/'

    # Setup HTTP session
    session = requests.Session()
    session.auth = (username, password)   # User credential
    session.verify = False                # No Certificate verification

    db_links = get_links_from_jgs(linksurl, session)

    if mode == 'O':
        output_to_file(db_links, filename, filetype)
    else:
        jg_links = Links()
        jg_links.set_links(get_links_from_file(filename, filetype))
        if jg_links.validate_links():
            # Get userObjectId from system associate with request user
            jg_links.set_userobjectid(get_userobjectid(session, usersurl))
            if jg_links.get_userobjectid() != '':
                if mode == 'I':
                    jg_links.create_links(linksurl, session)
                else:
                    jg_links.modify_links(linksurl, session, db_links, mode)
